package ga.view.config;

import ga.core.validation.IValidationRule;
import ga.core.validation.IValidator;
import ga.core.validation.RuleValidator;

import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jdom.Element;

/**
 * This settings implementation parses and holds the settings for a GA.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class GASettings extends AbstractSettings {

  public static final String PRE_EVALUATION_STATE = "preEvaluationState";
  public static final String POST_EVALUATION_STATE = "postEvaluationState";
  public static final String EVALUATOR = "evaluator";

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(GASettings.class
      .getName());

  private Element element;
  private boolean hasRun;

  /**
   * Instantiates new GA settings.
   * 
   * @param element
   *          the element
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public GASettings(final Element element) {
    parse(element);
  }

  /**
   * Sets the hasRun flag.
   * 
   * @param hasRun
   *          the new hasRun flag.
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setHasRun(final boolean hasRun) {
    this.hasRun = hasRun;
  }

  /**
   * Checks if this has run before.
   * 
   * @return true, if successful
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public boolean hasRun() {
    return hasRun;
  }

  /**
   * Resets these settings by clearing all settings and reparsing the element.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void reset() {
    clear();
    parse(element);
    hasRun = false;
  }

  /**
   * Parses the settings from an xml element.
   * 
   * @param element
   *          the element
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  @SuppressWarnings({ "unchecked", "rawtypes" })
  public void parse(final Element element) {
    this.element = element;
    final Element valuesE = element.getChild("values", element.getNamespace());
    if (valuesE != null) {
      Iterator<?> it = valuesE.getChildren("simpleValue",
          element.getNamespace()).iterator();
      while (it.hasNext()) {
        final Element sValue = (Element) it.next();

        parseSimple(sValue);
      }

      it = valuesE.getChildren("objectValue", element.getNamespace())
          .iterator();
      while (it.hasNext()) {
        final Element oValue = (Element) it.next();

        parseObject(oValue);
      }
    }

    Element e = element.getChild("phenotypeGenerator", element.getNamespace());
    if (e != null) {
      parseObject(e, "phenotypeGenerator");
    }

    e = element.getChild("individualFactory", element.getNamespace());
    if (e != null) {
      parseObject(e, "individualFactory");
    }

    e = element.getChild(EVALUATOR, element.getNamespace());
    if (e != null) {
      parseObject(e, EVALUATOR);
    }

    e = element.getChild("population", element.getNamespace());
    if (e != null) {
      parseObject(e, "population");
    }

    e = element.getChild("selector", element.getNamespace());
    if (e != null) {
      parseObject(e, "selector");
    }

    e = element.getChild("validator", element.getNamespace());
    if (e != null) {
      final IValidator<?> v = (IValidator<?>) parseObject(e, "validator");

      if (v != null && v instanceof RuleValidator<?>) {
        final Iterator<?> it = e.getChildren("rule", element.getNamespace())
            .iterator();

        int i = 1;

        while (it.hasNext()) {
          final Element sValue = (Element) it.next();

          final IValidationRule r = (IValidationRule) parseObject(sValue,
              "rule" + i);

          if (r != null) {
            ((RuleValidator) v).addRule(r);

            i++;
          }
        }
      }
    }

    e = element.getChild("mutationOp", element.getNamespace());
    if (e != null) {
      parseObject(e, "mutationOp");
    }

    e = element.getChild("crossoverOp", element.getNamespace());
    if (e != null) {
      parseObject(e, "crossoverOp");
    }

    e = element.getChild("ga", element.getNamespace());
    if (e != null) {
      parseObject(e, "ga");
    }

    e = element.getChild(PRE_EVALUATION_STATE, element.getNamespace());
    if (e != null) {
      parseObject(e, PRE_EVALUATION_STATE);
    }

    e = element.getChild(POST_EVALUATION_STATE, element.getNamespace());
    if (e != null) {
      parseObject(e, POST_EVALUATION_STATE);
    }
  }

  /**
   * Parses a simple parameter from an element that contains a ground datatype
   * (boolean, string, int, long, float, double).
   * 
   * @param sValue
   *          the Element
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void parseSimple(final Element sValue) {
    final String id = sValue.getAttributeValue("id");
    final String type = sValue.getAttributeValue("type");
    final String value = sValue.getAttributeValue("value");

    final Object o = createSimple(type, value);
    putUnique(id, o);
  }

  /**
   * Parses an object from xml and stores it with the id.
   * 
   * @param cValue
   *          the Element
   * @return the object
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private Object parseObject(final Element cValue) {
    final String id = cValue.getAttributeValue("id");
    return parseObject(cValue, id);
  }

  /**
   * Parses an object from xml and stores it with the given id.
   * 
   * @param cValue
   *          the Element
   * @param id
   *          The id to store it.
   * @return the object
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private Object parseObject(final Element cValue, final String id) {
    final int numParameters = cValue.getChildren("simpleParameter",
        cValue.getNamespace()).size()
        + cValue.getChildren("objectParameter", cValue.getNamespace()).size();
    final Object[] parameters = new Object[numParameters];
    final Class<?>[] parameterClasses = new Class<?>[numParameters];

    Iterator<?> it = cValue.getChildren("simpleParameter",
        cValue.getNamespace()).iterator();
    while (it.hasNext()) {
      final Element sValue = (Element) it.next();

      final int index = Integer.valueOf(sValue.getAttributeValue("index"));
      final String id2 = sValue.getAttributeValue("id");

      final Object o = get(id2);

      parameters[index] = o;
      parameterClasses[index] = o.getClass();
    }

    it = cValue.getChildren("objectParameter", cValue.getNamespace())
        .iterator();
    while (it.hasNext()) {
      final Element sValue = (Element) it.next();

      final int index = Integer.valueOf(sValue.getAttributeValue("index"));
      final String id2 = sValue.getAttributeValue("id");

      final Object o = get(id2);

      parameters[index] = o;
      parameterClasses[index] = o.getClass();
    }

    final String clss = cValue.getAttributeValue("class");

    // for first, ignore parameters
    try {
      final Object o = createObject(clss, parameterClasses, parameters);

      putUnique(id, o);
      return o;

    } catch (final Exception e) {
      LOGGER.log(Level.SEVERE, "Failed to parse object of class " + clss, e);
    }

    return null;
  }

}
